/**
* \file: eapSocket.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <iomanip>
#include <pthread.h>
#include <arpa/inet.h>
#include <errno.h>
#include <string.h>
#include "CCarLifeLib.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "eapSocket.h"
#include "CCarLifeLog.h"
// all the logging messages in this file are replaced to DLT by ADIT
LOG_IMPORT_CONTEXT(bdcl_core) //added by ADIT
#define EAP_HEADER_SIZE           8//may change
#define EAP_RECEIVER_THREAD_PRIO 61

using namespace std;


eapSocket::eapSocket()
{
    receiverThreadTerminationFlag = false;
    gotHeader  = false;
    msgSize    = 0;
    mWriteFd   = -1;
    mReadFd    = -1;
    mChannelId = -1;
    receiverThreadId = 0;
}

eapSocket::~eapSocket()
{

    //closing EA native Read and Write Endpoints
    if(mWriteFd >= 0){
        LOGD_DEBUG((bdcl_core,"Closing write end point"));
        close(mWriteFd);
    }
    if(mReadFd >= 0){
        LOGD_DEBUG((bdcl_core,"Closing read end point"));
        close(mReadFd);
    }

}

bool eapSocket::connect(const int inWrId, const int inRdId)
{
    gotHeader  = false;
    msgSize    = 0;
    mChannelId = -1;

    int err = 0;
    int rc = 0;

    mWriteFd = inWrId;
    mReadFd = inRdId;

    if ((mWriteFd >= 0) && (mReadFd >= 0))
    {
        LOGD_DEBUG((bdcl_core,"%s() EA Native EndPoints Opened Successfully", __FUNCTION__));

        if (createSocketRingBuffers()) {
            LOGD_DEBUG((bdcl_core,"%s() Creating the Ring Buffers Sucessfull", __FUNCTION__));
        } else {
            LOG_ERROR((bdcl_core,"%s() Creating the Ring Buffers Failed", __FUNCTION__));
            return false;
        }

        pthread_create(&receiverThreadId, NULL, &eapSocket::_WorkerThread, this);
        if (err)
        {
            LOG_ERROR((bdcl_core,"%s() Error in creating the recievePacket thread", __FUNCTION__));
            return false;
        } else {
            LOGD_DEBUG((bdcl_core,"%s() Creating recieverPacket thread successfull", __FUNCTION__));
        }

        rc = pthread_setname_np(receiverThreadId,"eapReceiver");
        if (rc != 0)
            LOG_WARN((bdcl_core,"%s() Error in setting thread Name for recieverThread", __FUNCTION__));

        int sched_policy;
        struct sched_param sched_param;

        err = pthread_getschedparam(receiverThreadId, &sched_policy, &sched_param);
        if (err == 0)
        {
            sched_param.sched_priority = EAP_RECEIVER_THREAD_PRIO;
            err = pthread_setschedparam(receiverThreadId, SCHED_FIFO, &sched_param);
            if (err != 0)
            {
                LOG_ERROR((bdcl_core, "Failed to set eapReceiver thread priority with error %d", err));
            }
        }
        else
        {
            LOG_ERROR((bdcl_core, "Failed to get eapReceiver thread priority with error %d", err));
        }
    }
    else
    {
        LOGD_DEBUG((bdcl_core,"%s() EA Native EndPoints are invalid (mWriteFd: %d, mReadFd: %d)",
                __FUNCTION__, mWriteFd, mReadFd));
        return false;
    }


    return true;
}


#ifdef DUMP_SEND_TRAFFIC

static void hexdump(const unsigned char * data, int len)
{
    for(int i = 0;i<len;i++)
    {
        printf("0x%x ", data[i]);
    }

    printf("\n");
}
#else
static void hexdump(const unsigned char * data, int len)
{
    (void) data;
    (void) len;
}
#endif

struct __attribute__((__packed__)) eaHeader {
    u32 channelId;
    u32 msgSize;
};


bool eapSocket :: sendEapData(const unsigned char * data, u32 size, int channelId)
{
    struct eaHeader hdr;
    bool ret = false;

    LOGD_VERBOSE((bdcl_core,"%s() Channel Id = %d and number of bytes = %d", __FUNCTION__, channelId, size));

    hdr.channelId = htonl(channelId);
    hdr.msgSize = htonl(size);

    hexdump((const unsigned char *)&hdr, sizeof(hdr));
    int writeResult = write(mWriteFd, (const unsigned char *)&hdr, sizeof(hdr));
    if (writeResult != sizeof(hdr))
    {
        LOG_ERROR((bdcl_core,"%s() Sending eapHeader failed: %d", __FUNCTION__, writeResult));
        return false;
    }

    hexdump((const unsigned char *)data,size);
    writeResult = write(mWriteFd, data, size);

    if ((writeResult > 0) && ((uint32_t)writeResult == size))
    {
        LOGD_VERBOSE((bdcl_core, "%s() Data writeResult: %d",__FUNCTION__, writeResult));
        ret = true;
    }
    else
    {
        LOG_ERROR((bdcl_core,"%s() Sending data failed: %d", __FUNCTION__, writeResult));
        ret = false;
    }

    return ret;
}


// TBD: error logging:

bool eapSocket :: send(u8* data, u32 len, int inChannelId) {

    std::unique_lock<std::mutex> lck(sendPacketsLock);

    if ( mChannelId != -1 && inChannelId != mChannelId )
    {
        LOGD_DEBUG((bdcl_core,"%s: transfer on channel %d on going, blocking channel %d",__FUNCTION__, mChannelId,inChannelId));
        do
        {
            channelVacant.wait(lck);
        } while(mChannelId != -1);

        LOGD_DEBUG((bdcl_core,"%s: unblocking channel %d",__FUNCTION__, inChannelId));
    }

    if(!gotHeader)
      {
        if(len < sizeof(uint16_t))
          {
            LOG_ERROR((bdcl_core,"%s, incoming header too small: %d", __FUNCTION__, len ));
            return false;
        }

        uint32_t size = 0;
        if ((CMD_CHANNEL == inChannelId) || (CTRL_CHANNEL == inChannelId))
        {
            size = ntohs(*((uint16_t*)data));
        }
        else
        {
            size = ntohl(*((uint32_t*)data));
        }

        if (0 == size)
        {
            // we got a header for a message with 0 byte payload. Therefore we can
            // send the message immediately
            return sendEapData(data, len, inChannelId);
        }
        else
        {
            // store the header and send it together with the payload which will
            // be received by this function with the next call
            if (len > sizeof(sendMsg))
            {
                LOG_ERROR((bdcl_core,"%s: message too large: %d. Rejecting", __FUNCTION__, len));
                return false;
            }

            msgSize = len;
            memcpy(&sendMsg[0],data,len);
            gotHeader = true;
            mChannelId = inChannelId;

            return true;
        }
    }

    if (len + msgSize > sizeof(sendMsg))
    {
        LOG_ERROR((bdcl_core," %s: message too large: (%d + %d > %d). Rejecting",
                __FUNCTION__, len, msgSize, sizeof(sendMsg)));

        gotHeader = false;
        mChannelId = -1; // transfer on channel done.
        lck.unlock();
        channelVacant.notify_one();

        return false;
    }

    memcpy(&sendMsg[msgSize],data,len);
    msgSize += len;

    gotHeader = false;
    bool ret = sendEapData(&sendMsg[0], msgSize, inChannelId);


    mChannelId = -1; // transfer on channel done.
    lck.unlock();
    channelVacant.notify_one();

    return ret;
}

void eapSocket::recievePackets() {

    CCarLifeLog::carLifeLog("recieve packet thread\n");
    struct eaHeader *hdr;

    int portNo = 0 ;
    int msgLen = 0 ;
    int readResult = 0 ;
    unsigned char* readPtr  = NULL ;
    unsigned char* msgBegin = NULL ;
    unsigned char* msgEnd   = NULL ;
    unsigned char  inBuf[1024*1024];
    unsigned int   lenInBuf = 1024*1024;
    unsigned int   readLen  = 16*1024;//read works only for some magic numbes 4k,8k,16k..etc
    unsigned char* bufPtr   = NULL;
    unsigned int   totalRead = 0;


    while(receiverThreadTerminationFlag == false)
    {
        bufPtr = NULL;
        readResult = 0;
        totalRead = 0;

        bufPtr = inBuf;
        // 1-read all available data first from the end point
        // Its currently assumed that, we are not receiving more
        // data than the local buffer size [1024*1024] 1MB
        do
        {
            readResult = read(mReadFd, bufPtr, readLen);
            if (readResult > 0)
            {
                bufPtr = bufPtr + readResult;
                totalRead = totalRead + readResult ;
            }
            else
            {
                LOG_WARN((bdcl_core,"%s() EAP_read read result 0 or failed: %d errorno:%d:%s\n",
                    __PRETTY_FUNCTION__, readResult, errno, strerror(errno)));
                bufPtr = NULL;
                readResult = 0;
                bufPtr = inBuf;
            }

        } while(readResult == static_cast<int> (readLen)); // todo warning: comparison between signed and unsigned integer expressions

        //2-Parse receive data and push to respective channels
        if (totalRead <= 0)
        {
            LOG_ERROR((bdcl_core,"%s() EAP_read failed: %d errorno:%d:%s\n",
                    __PRETTY_FUNCTION__, readResult, errno, strerror(errno)));

            //todo:handle errno and terminate thread accordingly
            //receiverThreadTerminationFlag = false;
        }
        else
        {
            msgBegin = inBuf;
            readPtr  = msgBegin;

            if (totalRead >= EAP_HEADER_SIZE)
            {
                msgEnd   = msgBegin + totalRead;

                while(readPtr < msgEnd)
                {
                    if(readPtr < msgBegin)
                    {
                        LOG_ERROR((bdcl_core,"%s() EAP_read failed readPtr is broken\n",
                                __PRETTY_FUNCTION__));
                        break;
                    }
                    hdr = (struct eaHeader *)readPtr;
                    portNo = ntohl(hdr->channelId);
                    msgLen = ntohl(hdr->msgSize) ;

                    if((portNo < 0) || (msgLen < 0))
                    {
                        LOG_ERROR((bdcl_core,"%s() EAP_read failed portNo or msgLen is broken\n",
                                __PRETTY_FUNCTION__));
                        break;
                    }

                    readPtr = readPtr + EAP_HEADER_SIZE;

                    pushData(portNo, msgLen, readPtr);
                    readPtr = readPtr + msgLen;
                }
            }
        }
    }
    LOGD_DEBUG((bdcl_core,"%s() Thread Terminated", __PRETTY_FUNCTION__));
}


void eapSocket::wakeupThreads()
{
    //terminate recievePackets Thread
    receiverThreadTerminationFlag = true;
    pthread_join(receiverThreadId, nullptr);

    //Terminate channel threads
    for(int i=0;i<MAX_CHANNEL;i++)
    {
        socketRingBuffer[i].terminate();
    }
}


bool eapSocket::pushData(const u8 inChannelId, const int inTotalRecvMessLength, unsigned char* readPtr)
{
    if (socketRingBuffer[inChannelId-1].writeToRingBuffer(readPtr,
            inTotalRecvMessLength)) {
        LOGD_VERBOSE((bdcl_core,"%d bytes of Data is pushed sucessfully",inTotalRecvMessLength));
    }
    else
    {
        LOG_ERROR((bdcl_core,"%d bytes of Data is pushed to %s ring buffer Failed",inTotalRecvMessLength, ringBufferNames[inChannelId-1]));
        return false;
    }
    return true;
}

bool eapSocket::getData(u32 inLen, u8 inChannelId, u8* outData)
{
    if(socketRingBuffer[inChannelId-1].readFromRingBuffer(outData, inLen))
    {
        LOGD_VERBOSE((bdcl_core, "eapSocket::getData() %d bytes of Data is read sucessfully", inLen));
        return true;
    }
    else
    {
        LOG_INFO((bdcl_core, "eapSocket::getData() %d bytes of Data from %s ring buffer read Failed", inLen, ringBufferNames[inChannelId-1]));
        return false;
    }
}


bool eapSocket::createSocketRingBuffers()
{

    bool rc = false;

    for (int i = 0; i < MAX_CHANNEL; i++)
    {
        switch(i)
        {
        case 0:
            rc = socketRingBuffer[i].allocateBuffer(CMD_BUF_SIZE);
            break;
        case 1:
            rc = socketRingBuffer[i].allocateBuffer(VIDEO_BUF_SIZE);
            break;
        case 2:
            rc = socketRingBuffer[i].allocateBuffer(MEDIA_BUF_SIZE);
            break;
        case 3:
            rc = socketRingBuffer[i].allocateBuffer(TTS_BUF_SIZE);
            break;
        case 4:
            rc = socketRingBuffer[i].allocateBuffer(VR_BUF_SIZE);
            break;
        default:
            break;
        }

        if (!rc)
        {
            LOG_ERROR((bdcl_core, "error in creating the RingBuffer: %s",ringBufferNames[i]));
            return false;
        } else
        {
            LOGD_DEBUG((bdcl_core,"%s RingBuffer created successfully",ringBufferNames[i]));
        }
    }
    return true;
}


int eapSocket::convertLength(){
    return (headerBuffer[4] << 24 | headerBuffer[5] << 16 | headerBuffer[6] << 8
            | headerBuffer[7]);
}

